using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Reflection;
using System.Threading;
using SDRSharp.FrequencyManager;
using SDRSharp.Common;
using SDRSharp.PanView;
using SDRSharp.Radio;

namespace SDRSharp.Trunker
{
    [DesignTimeVisible(true)]
    [Category("SDRSharp")]
    [Description("Trunker Control Panel")]
    public unsafe partial class TrunkerPanel : UserControl
    {
        private readonly SettingsPersisterTrunker _settingsPersister;
        private List<TrunkerSettings> _trunkerConfig;
        private TrunkerLogger _trunkLoggerData;
        private ISharpControl _controlInterface;
        private TrunkerPlugin _plugin;
        private FileSystemWatcher _fileWatcher;
        private readonly object _fftResolutionLockObject = new object();
        private string _trunkingFileName = "sdrsharptrunking.log";
		//private byte[] _fftSpectrum; //byte type is obsolete
		private float[] _fftSpectrum;
        private const int MUTE = 15;
        private int _cachedAudioGain = MUTE;
        private System.Timers.Timer _retuneDelay = null;
        private int _beepDelay;
        private bool unitrunkerConfigured = false;
        private bool loggingConfigured = false;
        private bool loggingEnabled = false;
        private bool loggingSystemMessageDisplayed = false;
        private bool unitrunkerConfigureMessageDisplayed = false;
        private bool initializing = true;
        private String _version;
        private bool unitrunkerIsParked = false;
        private bool _sdrsharp_rev_1133 = false;
        #if DEBUG
            private bool debuggingLogSystem = false;
        #endif

        public TrunkerPlugin PlugIn
        {
            get { return _plugin; }
            set { _plugin = value; }
        }
        
        public TrunkerPanel(ISharpControl control, TrunkerPlugin _parentplug)
        {
            InitializeComponent();
            _version = Assembly.GetExecutingAssembly().GetName().Version.ToString() + " ZefieMod";
            #if DEBUG
                _version += " DEBUG";
            #endif
            this._plugin = _parentplug;
            _controlInterface = control;
            _cachedAudioGain = _controlInterface.AudioGain;
            _controlInterface.PropertyChanged += new PropertyChangedEventHandler(_controlInterface_PropertyChanged);
            //_fftSpectrum = new byte[_controlInterface.FFTResolution];
			_fftSpectrum = new float[_controlInterface.FFTResolution];
            _settingsPersister = new SettingsPersisterTrunker();
            _trunkerConfig = _settingsPersister.readConfig();
            _trunkLoggerData = new TrunkerLogger();
            setVersion();
            this._sdrsharp_rev_1133 = isRev1133orBetter;
            ProcessSavedSettings();
            initializing = false;
            
            #if DEBUG
                // Only run this if debuggingLogSystem = true, and we are in debug mode
                if (debuggingLogSystem)
                    spawnLogDialog_Click(this, null);
            #endif
        }

        private void setVersion()
        {
            this.versionAndStatus.Text = "v" + _version;
        }

        private bool isRev1133orBetter
        {
            get
            {
                if (_controlInterface.GetType().GetProperty("SourceIsTunable") != null)
                {
                    return true;
                }
                return false;
            }
        }

        private bool isRev1134orBetter
        {
            get
            {
                if (_controlInterface.GetType().GetProperty("FFTRange") != null && _controlInterface.GetType().GetProperty("FFTOffset") != null)
                {
                    return true;
                }
                return false;
            }
        }

        private bool isSourceTunable
        {
            get
            {
                return (bool)_controlInterface.GetType().GetProperty("SourceIsTunable").GetValue(_controlInterface, null);
            }
        }

        private float getFFTRange
        {
            get
            {
                return (float)_controlInterface.GetType().GetProperty("FFTRange").GetValue(_controlInterface, null);
            }
        }

        private float getFFTOffset
        {
            get
            {
                return (float)_controlInterface.GetType().GetProperty("FFTOffset").GetValue(_controlInterface, null);
            }
        }

        void ProcessSavedSettings()
        {
            // There is probably a cleaner way, but right now I'm going for functionality.
            try
            {
                // This will fail if we have no existing configuration
                this.uniTrunkerHomeDirectory.Text = _trunkerConfig[0].unitrunkerPath;
                this.singleLogFilePath.Text = _trunkerConfig[0].logPath;
                this.muteEdacsBeep.Checked = _trunkerConfig[0].isMuteEDACSEnabled;
                this.muteWhileParkedCheckbox.Checked = _trunkerConfig[0].isMuteControlEnabled;
                this.trunkingControlChannelFreq.Value = (long)_trunkerConfig[0].controlFreq;
                this.retuneDelayMethod.SelectedIndex = (int)_trunkerConfig[0].delayRetuneMethod;
                this.retuneMinSignalLevel.Value = (decimal)_trunkerConfig[0].delayRetuneMinimumSignal;
                this.stickyTuneTrunkingCheckbox.Checked = _trunkerConfig[0].isDelayRetuneEnabled;

                if (this.singleLogFilePath.Text != null)
                {
                    this.loggingConfigured = true;
                    this.singleLogFileEnabled.Checked = _trunkerConfig[0].isLoggingEnabled;
                }

                if (this.uniTrunkerHomeDirectory.Text != null)
                {
                    this.unitrunkerConfigured = true;
                    this.trunkingEnabledCheckbox.Checked = _trunkerConfig[0].isEnabled;
                }
            }
            catch (Exception ex)
            {
                // We have no existing config, which is fine, but we need to catch the exception
                // that _trunkerConfig[0] is an invalid index
                DoNothing(ex);
            }
        }

		private bool checkUnitrunkerPath(String path)
		{
            if( Directory.Exists(path) )
            {
                // The directory is defined, and it exists, but does it have Unitrunker and Remote.dll?
                if( File.Exists(path + "\\Unitrunker.xml") )
                {
                    // Unitrunker executable was found
                    if( File.Exists(path + "\\Remote.dll") )
                    {
                        // So was Remote.dll, we are okay now.
                        unitrunkerConfigured = true;
                        return true;
                    }
                    else
					{
                        MessageBox.Show(@"Warning: Remote.dll not found in UniTrunker configuration directory.  Copy Remote.dll from SDRSharp directory to directory that contains UniTrunker.xml (typically c:\users\[username]\AppData\Roaming\UniTrunker).");
                    }
                }
                else 
                {
                    MessageBox.Show("Warning: UniTrunker.xml configuration file not found, ensure that the directory selected is the 'AppData' directory for unitrunker");
                }
            }
            return false;
        }

        void WriteConfig()
        {
            // There is probably a cleaner way, but right now I'm going for functionality.

            // This is basically the opposite of ProcessSavedSettings().
            // It shouldn't really matter what order we save these.

            // We are going to try to read from _trunkerConfig[0]. If we are successful, it exists
            // and we can write the settings. If not, we need to catch the exception and create the index.
            if( !initializing )
            {
                // don't write the config when we are initializing.
                // WriteConfig() will be fired when we change the checkbox due to OnChange
                // We want this but not during initialization.
                try
                {
                    _trunkerConfig[0].settingsExist();
                }
                catch (Exception ex)
                {
                    DoNothing(ex);
                    _trunkerConfig.Insert(0, new TrunkerSettings());
                }
                _trunkerConfig[0].isEnabled = this.trunkingEnabledCheckbox.Checked;
                _trunkerConfig[0].unitrunkerPath = this.uniTrunkerHomeDirectory.Text;
                _trunkerConfig[0].logPath = this.singleLogFilePath.Text;
                _trunkerConfig[0].isDelayRetuneEnabled = this.stickyTuneTrunkingCheckbox.Checked;
                _trunkerConfig[0].isMuteEDACSEnabled = this.muteEdacsBeep.Checked;
                _trunkerConfig[0].isMuteControlEnabled = this.muteWhileParkedCheckbox.Checked;
                _trunkerConfig[0].isLoggingEnabled = this.singleLogFileEnabled.Checked;
                _trunkerConfig[0].controlFreq = this.trunkingControlChannelFreq.Value;
                _trunkerConfig[0].delayRetuneMethod = this.retuneDelayMethod.SelectedIndex;
                _trunkerConfig[0].delayRetuneMinimumSignal = this.retuneMinSignalLevel.Value;
                _settingsPersister.writeConfig(_trunkerConfig);
            }
        }

        void _controlInterface_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if( e.PropertyName.Equals("AudioGain") )
			{
                _cachedAudioGain = _controlInterface.AudioGain;
            }
            else if( e.PropertyName.Equals("CenterFrequency") )
            {
                delayRetune();
            }
        }

        private bool isRetuningDelayed()
        {
            return _retuneDelay != null;
        }

        private bool HasMethod(object objectToCheck, string methodName)
        {
            var type = objectToCheck.GetType();
            return type.GetMethod(methodName) != null;
        }

        private bool shouldWeRetune
        {
            get
            {
                // Save a bit of CPU by not calculating if we don't need to
                if( this.stickyTuneTrunkingCheckbox.Checked )
                {
                    // Reference the config data rather than reading the SelectedIndex in realtime.

                    // This prevents this function from changing methods while the user is deciding
                    // as we only write the config when the user has decided (dropdown box is closed)

                    // Don't retune if uniTrunker isn't parked
                    if( !unitrunkerIsParked )
                    {
                        return false;
                    }

                    if( _trunkerConfig[0].delayRetuneMethod == 0 && _controlInterface.SquelchEnabled )
                    {
                        // Use Squelch
                        return !_controlInterface.IsSquelchOpen;
                    }
                    if( _trunkerConfig[0].delayRetuneMethod == 1 )
                    {
                        // Use Signal

                        // Put a thread lock on so that we don't run this more than once at a time
                        Monitor.Enter(_fftResolutionLockObject);
                        float signalLevel = getPeakSignal();
                        Monitor.Exit(_fftResolutionLockObject);

                        this.currentDbValue.Text = signalLevel.ToString() + "dB";
                        return (signalLevel < (float)this.retuneMinSignalLevel.Value);
                    }
                }
                else
                {
                    // Set curernt dB to "Off" because we are not tuned
                    // Check for the element to prevent crashing from a stupid VS bug...
                    if( this.currentDbValue != null )
                        this.currentDbValue.Text = "Off";
                }
                return false;
            }
        }

        private float getPeakSignal()
        {
            // attempts to determine the peak signal dB from the currently tuned frequency,
            // and within the currently defined filter bandwidth          
            lock (_fftResolutionLockObject)
            {
                if (_fftSpectrum.Length != _controlInterface.FFTResolution)
                {
                    //_fftSpectrum = new byte[_controlInterface.FFTResolution];
					_fftSpectrum = new float[_controlInterface.FFTResolution];
                }
                _controlInterface.GetSpectrumSnapshot(_fftSpectrum);
                int bw = _controlInterface.FilterBandwidth;
                int rfbw = _controlInterface.RFBandwidth;
                long freq = _controlInterface.Frequency;
                long minFrequency = freq - (bw / 2);
                long maxFrequency = freq + (bw / 2);
                float peak = -120000; // testing
                float current = 0;
                float hzPerBin = (rfbw / _fftSpectrum.Length);
                for (long i = minFrequency; i <= maxFrequency; i = (i + (long)hzPerBin))
                {
                    current = _fftSpectrum[convertFrequencyToBin(i)];
                    if (current > peak)
                    {
                        peak = current;
                    }
                }

                return (float)Math.Round(-((130 / 255.0f) * (255 - peak)),1);
            }
        }

        private int convertFrequencyToBin(long frequency)
        {
            //converts frequency to fft bin
            int bin = 0;
            long center = _controlInterface.CenterFrequency;
            int bandwidth = _controlInterface.RFBandwidth;
            long minFrequency = center - (bandwidth / 2);
            long maxFrequency = center + (bandwidth / 2);
            if (frequency > minFrequency && frequency < maxFrequency)
            {
                float hzPerBin = bandwidth / _controlInterface.FFTResolution;
                bin = (int)((frequency - minFrequency) / hzPerBin);
            }
            return bin;
        }

        private void doTrunkTuneTimerLoop()
        {
            if (!stickyTuneTrunkingCheckbox.Checked || //if we are not delaying retune based on signal strength or
                !isRetuningDelayed() ||                //if retuning isn't delayed due to tuner latency
                                                        //or if (we are delaying retune and) the current signal level is below the threshold
                !this.shouldWeRetune)
            {
                // debug
                    // if the plugin is tuning to the control channel and there is a value there
                    if (trunkingTuneToControlChannel.Checked && trunkingControlChannelFreq.Value > 0 )
                    {
                        // Are we running SDRSharp svn revision 1133 or better?
                        if (_sdrsharp_rev_1133)
                        {
                            if (isSourceTunable)
                            {
                                _controlInterface.CenterFrequency = (long)trunkingControlChannelFreq.Value - 100000;

                            }
                        }
                        else
                        {
                            // We are running 1132 or older, so lets fall back
                            // It will display that nasty error message when used with Soundcard or WAV, but at least it'll run...
                            try
                            {
                                _controlInterface.CenterFrequency = (long)trunkingControlChannelFreq.Value - 100000;

                            }
                            catch (Exception f)
                            {
                                DoNothing(f);
                            }
                        } 
                        _controlInterface.Frequency = (long)trunkingControlChannelFreq.Value;
                        _controlInterface.AudioGain = _cachedAudioGain;
                        delayRetune(); //delay retuning
                    }
                    else
                    {
                        //tuneToTrunkingFile();
                    }
            }
        }

        private void WriteLog(String txtToWrite, string filePath)
        {
            // Are the occasional crashs related to trying to write the file too fast?
            try
            {
                using (StreamWriter outfile = new StreamWriter(filePath))
                {
                    outfile.Write(txtToWrite.ToString());
                }
            }
            catch (Exception e)
            {
                DoNothing(e);
            }

        }

        private void writeSingleLog()
        {
            if (singleLogFilePath.Text != null && this.loggingEnabled)
            {
                String outstring = LogParser.ParseLogStyle(this._trunkLoggerData);
                if (outstring != null)
                {
                    versionAndStatus.Text = outstring;
                    WriteLog(outstring, singleLogFilePath.Text);
                }
            }
        }

        private void PrepareLog()
        {
            // Wipe the log data to prepare for new
            _trunkLoggerData = null;
            _trunkLoggerData = new TrunkerLogger();
        }

        private void getLogData()
        {
            PrepareLog();
            try
            {
                using (StreamReader log = new StreamReader(uniTrunkerHomeDirectory.Text + @"\" + _trunkingFileName))
                {
                    string line;
                    string[] row;
                    while (((line = log.ReadLine()) != null))
                    {
                        //spectrumAnalyzer.Title = line;
                        try
                        {   //for logging
                            row = line.Split('\t');
                            if ("frequency".Equals(row[2]))
                            {
                                // we are on the first line, we need the second line;
                                continue;
                            }
                            var freq = Convert.ToDecimal(row[2]);
                            int rowcnt = row.Count();
                            _trunkLoggerData.currentFrequency = freq;

                            if (rowcnt >= 1)
                                _trunkLoggerData.currentAction = row[0];

                            if (rowcnt >= 2)
                                _trunkLoggerData.currentReceiver = row[1];

                            if (rowcnt >= 4)
                                _trunkLoggerData.currentTrunkgroup = row[3];

                            if (rowcnt >= 5)
                                _trunkLoggerData.currentTrunklabel = row[4];

                            if (rowcnt >= 6)
                                _trunkLoggerData.currentSourcegroup = row[5];

                            if (rowcnt >= 7)
                                _trunkLoggerData.currentSourcelabel = row[6];


                            if ("Park".Equals(_trunkLoggerData.currentAction))
                                unitrunkerIsParked = true;
                            else
                                unitrunkerIsParked = false;
                        }
                        catch (Exception e)
                        {
                            DoNothing(e);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                DoNothing(e);
            }

        }

        private void tuneToTrunkingFile()
        {
                if ("Listen".Equals(_trunkLoggerData.currentAction) || "Park".Equals(_trunkLoggerData.currentAction))
                {
                    if (singleLogFileEnabled.Checked)
                    {
                        writeSingleLog();
                    }
                    
                    // Are we running SDRSharp svn revision 1133 or better?
                    if (_sdrsharp_rev_1133)
                    {
                        if (isSourceTunable)
                        {
                            _controlInterface.CenterFrequency = (long)_trunkLoggerData.currentFrequency - 100000;
                        }
                    }
                    else {
                        // We are running 1132 or older, so lets fall back
                        // It will display that nasty error message when used with Soundcard or WAV, but at least it'll run...
                        try
                        {
                            _controlInterface.CenterFrequency = (long)_trunkLoggerData.currentFrequency - 100000;
                        }
                        catch (Exception f)
                        {
                            DoNothing(f);
                        }
                    }
                    _controlInterface.Frequency = (long)_trunkLoggerData.currentFrequency;
                    delayRetune();
                    if (muteWhileParkedCheckbox.Checked)
                    {
                        if ("Listen".Equals(_trunkLoggerData.currentAction))
                        {
                            _controlInterface.AudioGain = _cachedAudioGain;
                        }
                        else if ("Park".Equals(_trunkLoggerData.currentAction))
                        {
                            _controlInterface.AudioGain = MUTE;
                        }
                    }
                    //logfreq.Freq = (int)freq;
                    //_onPeak = true;
                    //if (_freqDb == null)
                    //{
                    //    _freqDb = new Dictionary<int, Frequency>();
                    //}
                    ////turrible hack but the whole thing is a throw away anyway
                    //_freqDb[logfreq.Freq] = logfreq;
                }
                //if (!String.IsNullOrEmpty(row[4]))
                //{
                //    logfreq.Name = row[4];
                //}
                //setSpectrumTitle();
        }

        private void DoNothing(Exception ex)
        {
            // it is kinda pointless, and a waste of space, but it shuts up the VS Warnings for unused variables
        }

        private void initTrunkingFileWatcher()
        {
            if (_fileWatcher == null)
            {
                _fileWatcher = new FileSystemWatcher();
                _fileWatcher.Path = uniTrunkerHomeDirectory.Text;
                _fileWatcher.NotifyFilter = NotifyFilters.LastWrite;
                _fileWatcher.Filter = _trunkingFileName;
                _fileWatcher.Changed += new FileSystemEventHandler(OnChanged);
            }
            _fileWatcher.EnableRaisingEvents = true;
        }

        private void OnChanged(object source, FileSystemEventArgs e)
        {
            // Specify what is done when a file is changed, created, or deleted.
            getLogData();
            if (trunkingEnabledCheckbox.Checked && (trunkingControlChannelFreq.Value == _controlInterface.Frequency || (!stickyTuneTrunkingCheckbox.Checked || this.shouldWeRetune)))
            {
                tuneToTrunkingFile();
            }
        }

        private void trunkingEnabledCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (unitrunkerConfigured)
            {
                // We passed the check for finding Unitrunker and Remote.dll, so we resume normal operation
                if (trunkingEnabledCheckbox.Checked)
                {
                    trunkingUnitrunkerBrowseButton.Enabled = false;
                    _plugin.Bypass = false;
                    if (!String.IsNullOrEmpty(uniTrunkerHomeDirectory.Text))
                    {
                        initTrunkingFileWatcher();
                        try
                        {
                            getLogData();
                            if (trunkingEnabledCheckbox.Checked && (trunkingControlChannelFreq.Value == _controlInterface.Frequency || (!stickyTuneTrunkingCheckbox.Checked || this.shouldWeRetune)))
                            {
                                tuneToTrunkingFile();
                            }
                            else
                            {
                                writeSingleLog();
                            }
                        }
                        catch (Exception f)
                        {
                            DoNothing(f);
                        }
                    }
                }
                else
                {
                    // destroy all the things
                    if (_fileWatcher != null)
                    {
                        _fileWatcher.EnableRaisingEvents = false;
                        _fileWatcher.Dispose();
                        _fileWatcher = null;
                    }
                    if (_retuneDelay != null)
                    {
                        _retuneDelay.Stop();
                        _retuneDelay.Dispose();
                        _retuneDelay = null;
                    }                                        
                    _plugin.Bypass = true;
                    unitrunkerIsParked = false;
                    trunkingUnitrunkerBrowseButton.Enabled = true;
                    setVersion();
                }

                trunkingTimer.Enabled = trunkingEnabledCheckbox.Checked;
                WriteConfig();
            }
            else
            {
                if (unitrunkerConfigureMessageDisplayed == false)
                {
                    // The checkbox is still disabled because we don't know where Unitrunker is
                    MessageBox.Show("Please set the Unitrunker directory below, and verify Remote.dll is in the Unitrunker directory.", "Please set Unitrunker Location", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    unitrunkerConfigureMessageDisplayed = true; // this prevents a double message
                    trunkingEnabledCheckbox.Checked = false;
                }
                else
                {
                    // If the user clicks it again, we will display the error again.
                    unitrunkerConfigureMessageDisplayed = false;
                }
            }
        }

        private void uniTrunkerHomeDirectory_TextChanged(object sender, EventArgs e)
        {
            if (checkUnitrunkerPath(this.uniTrunkerHomeDirectory.Text)) {
                initTrunkingFileWatcher();
            }
        }

        private void trunkingLogFileButton_Click(object sender, EventArgs e)
        {
            SaveFileDialog fileDialog = new SaveFileDialog();
            fileDialog.FileName = "single.log";
            fileDialog.DefaultExt = "log";
            fileDialog.Filter = "Log File|*.log|Comma Seperated Value File|*.csv|Standard Text File|*.txt";
            if (fileDialog.ShowDialog() == DialogResult.OK)
            {
                singleLogFilePath.Text = fileDialog.FileName;
                loggingConfigured = true;
                WriteConfig();
            }
        }

        private void trunkingUnitrunkerBrowseButton_Click(object sender, EventArgs e)
        {
        BrowseForUnitrunker:
            FolderBrowserDialog folderDialog = new FolderBrowserDialog();

            if (folderDialog.ShowDialog() == DialogResult.OK)
            {
                if (checkUnitrunkerPath(folderDialog.SelectedPath))
                {
                    uniTrunkerHomeDirectory.Text = folderDialog.SelectedPath;
                    WriteConfig();
                }
                else
                {
                    goto BrowseForUnitrunker;
                }
            }
        }

        private void trunkingTimer_Tick(object sender, EventArgs e)
        {
            doTrunkTuneTimerLoop();
        }

        private void trunkingTuneToControlChannel_CheckedChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }

        private void delayRetune()
		{
			if (this._retuneDelay == null)
			{
				this._retuneDelay = new System.Timers.Timer(1250);
				this._retuneDelay.Elapsed += new System.Timers.ElapsedEventHandler(_retuneDelay_Elapsed);
				this._retuneDelay.Start();
			}
		}

        void _retuneDelay_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            lock (this._retuneDelay)
            {
                if (this.shouldWeRetune && trunkingEnabledCheckbox.Checked)
                {
                    _retuneDelay.Stop();
                    _retuneDelay.Dispose();
                    _retuneDelay = null;
                    getLogData();
                    tuneToTrunkingFile();
                }
            }
        }

        public void MuteBeeps(float* audioBuffer, int length)
        {
            string tmpfldr = System.Environment.GetEnvironmentVariable("TEMP");
            if (tmpfldr.Length <= 0)
            {
                tmpfldr = @"C:\temp";
            }
            var _logFile = new System.IO.StreamWriter(tmpfldr+@"\beeps6.csv", true);
            // nearestStep, freqName, freq,(1 - (nearestStep / freq)) * 1000000, _currentPeakStartTime, DateTime.Now.Subtract(_currentPeakStartTime).TotalSeconds, _currentPeakMaxVal, _currentPeakAvgVal);
            if (muteEdacsBeep.Checked)
            {
                int pass = 1200;
                //edacs beep detection and mutiong
                for (int i = 0; i < length - pass; )
                {
                    if (_beepDelay-- < 0)
                    {
                        _controlInterface.AudioGain = 30;
                    }
                    UnsafeBuffer buf = UnsafeBuffer.Create(pass, sizeof(float));
                    float* bufPtr = (float*)buf;
                    for (var j = 0; j < pass && i < length; i++, j++)
                    {
                        bufPtr[j] = audioBuffer[i * 2];
                    }
                    var samplerate = 48000;//_controlInterfacePanel.goertzelFreqNumeric.Value;
                    var beep = Goertzel.GetMag(Convert.ToInt32(samplerate), 4800, bufPtr, pass);
                    var beep2 = Goertzel.GetMag(Convert.ToInt32(samplerate), 3000, bufPtr, pass);
                    _logFile.WriteLine(String.Format("{0},{1},{2}",beep,beep2,i));
                    if (beep > .01)
                    {
                        _controlInterface.AudioGain = MUTE;
                        _beepDelay = 2;
                    }
                }
            }
            _logFile.Flush();
            _logFile.Close();
        }

        private void stickyTuneTrunkingCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            WriteConfig();
            if (!stickyTuneTrunkingCheckbox.Checked && trunkingEnabledCheckbox.Checked)
            {
                tuneToTrunkingFile();
            }
        }

        private void singleLogFileEnabled_CheckedChanged(object sender, EventArgs e)
        {
            // format to emulate
            // <talkgroup name> <frequency>MHz (newline)
            // end goal, options to allow
            // <talkgroup name>
            // <frequency>MHz
            // <talkgroup name> <source id>
            // <talkgroup name> <source id> <frequency>MHz

            if (loggingConfigured && singleLogFilePath.Text != null)
            {

                if (singleLogFileEnabled.Checked)
                {
                    // Enable the logger  
                    this.trunkingLogFileButton.Enabled = false;
                    this.loggingEnabled = true;
                    if (trunkingEnabledCheckbox.Checked)
                    {
                        if (_trunkLoggerData == null)
                            getLogData();
                        writeSingleLog();
                    }
                }
                else
                {
                    // Stop logging
                    this.trunkingLogFileButton.Enabled = true;
                    this.loggingEnabled = false;
                    versionAndStatus.Text = "v" + _version;
                }
                WriteConfig();
            }
            else
            {
                if (loggingSystemMessageDisplayed == false)
                {
                    MessageBox.Show("Please set the log file location.", "Where do I write the log?", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    loggingSystemMessageDisplayed = true; // this prevents a double message
                    singleLogFileEnabled.Checked = false;
                } 
                else
                {
                    // If the user clicks it again, we will display the error again.
                    loggingSystemMessageDisplayed = false;
                }
            }
        }

        private void trunkingControlChannelFreq_ValueChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }

        private void muteWhileParkedCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }

        private void muteEdacsBeep_CheckedChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }

        private void logFormatStyle_SelectedIndexChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }

        private void spawnLogDialog_Click(object sender, EventArgs e)
        {
            LogOptions form = new LogOptions();
            form.FormClosed += new FormClosedEventHandler(logStyleFormClosed);
            form.ShowDialog();
        }

        private void logStyleFormClosed(object sender, FormClosedEventArgs e)
        {
            _trunkerConfig = _settingsPersister.readConfig();
            if (singleLogFileEnabled.Checked)
                writeSingleLog();
        }

        private void retuneDelayMethod_SelectedIndexChanged(object sender, EventArgs e)
        {
            
            if (!this.retuneDelayMethod.DroppedDown && (this.retuneDelayMethod.SelectedIndex != this._trunkerConfig[0].delayRetuneMethod || initializing))
            {
                // only run if the user selected a value, rather than while the user is deciding...
                resizeForDelayMethod();
                WriteConfig();
            }
        }

        private void resizeForDelayMethod()
        {
            int resizeBy = 40;
            if (this.retuneDelayMethod.SelectedIndex == 0)
            {
                // Squelch mode, so hide the signal form and resize the elements
                // read the current value (which should be expanded) and contract
                this.retuneMinSignalLabel.Visible = false;
                this.retuneMinSignalLevel.Visible = false;
                this.currentDbLabel.Visible = false;
                this.currentDbValue.Visible = false;
                this.mainPanel.Height = this.mainPanel.Height - resizeBy;
                this.panelForOptionsAfterRetune.Top = this.panelForOptionsAfterRetune.Top - resizeBy;
            }
            if (this.retuneDelayMethod.SelectedIndex == 1)
            {
                // Signal mode, so show the signal form and resize the elements
                // read the current value (which should be contracted) and expand
                if (!initializing)
                {
                    // We don't move the elements while initializing, because the default layout is already set up for this in the designer.
                    this.mainPanel.Height = this.mainPanel.Height + resizeBy;
                    this.panelForOptionsAfterRetune.Top = this.panelForOptionsAfterRetune.Top + resizeBy;
                }
                this.retuneMinSignalLabel.Visible = true;
                this.retuneMinSignalLevel.Visible = true;
                this.currentDbLabel.Visible = true;
                this.currentDbValue.Visible = true;
            }
        }

        private void retuneMinSignalLevel_ValueChanged(object sender, EventArgs e)
        {
            WriteConfig();
        }
      }
}
